#!/usr/bin/php -q
<?php

#
# cacti script to read Motherboard Monitor stats with SNMP Informant
#
# sensor type OID values used for index values
# sensor types not matching requested type are ignored
# sensor entries with reading of "255" are ignored
# additional data is fetched as needed
#

#
# read in the available Cacti include files
#
if (file_exists(dirname(__FILE__) . "/../include/config.php")) {
	include_once(dirname(__FILE__) . "/../include/config.php");
}

if (file_exists(dirname(__FILE__) . "/../include/global.php")) {
	include_once(dirname(__FILE__) . "/../include/global.php");
}

if (file_exists(dirname(__FILE__) . "/../lib/snmp.php")) {
	include_once(dirname(__FILE__) . "/../lib/snmp.php");
}

#
# global declarations
#
global $sensor_type;
global $snmp_arguments;
global $snmp_hostname;
global $snmp_community;
global $snmp_version;
global $snmp_v3_username;
global $snmp_v3_password;
global $snmp_v3_authproto;
global $snmp_v3_privpass;
global $snmp_v3_privproto;
global $snmp_v3_context;
global $snmp_port;
global $snmp_timeout;
global $snmp_array;
global $cacti_request;
global $cacti_request_type;
global $cacti_request_index;
global $oid;
global $cmdoutput;
global $cmdoutputcount;
global $outputcount;

#
# make sure enough parameters were provided
#
if (count($_SERVER['argv']) < 12) {

	echo ("FAILED -- insufficient parameters provided\n");

	displaySyntax();

	exit(1);
}

#
# 1st parameter is always $sensor_type
#
$sensor_type = strtolower($_SERVER['argv'][1]);

if (($sensor_type != "fan") &&
	($sensor_type != "temperature") &&
	($sensor_type != "voltage")) {

	echo ("FAILED -- invalid sensor type\n");

	displaySyntax();

	exit(1);
}

#
# 2nd parameter is always $snmp_hostname
#
$snmp_hostname = substr($_SERVER['argv'][2],0,(strlen($_SERVER['argv'][2]) - 1));

#
# 3rd parameter is always $snmp_community; testing is done after we have the version
#
$snmp_community = substr($_SERVER['argv'][3],0,(strlen($_SERVER['argv'][3]) - 1));

#
# 4th parameter is always $snmp_version
#
$snmp_version = substr($_SERVER['argv'][4],0,(strlen($_SERVER['argv'][4]) - 1));

if (($snmp_version != 1) and ($snmp_version != 2) and ($snmp_version != 3)) {

	echo ("FAILED -- \"$snmp_version\" is not a valid SNMP version\n");

	displaySyntax();

	exit(1);
}

if (($snmp_version != 3) and ($snmp_community == "")) {

	echo ("FAILED -- invalid SNMP community string\n");

	displaySyntax();

	exit(1);
}

#
# 5th parameter is $snmp_v3_username
#
$snmp_v3_username = substr($_SERVER['argv'][5],0,(strlen($_SERVER['argv'][5]) - 1));

#
# 6th parameter is $snmp_v3_password
#
$snmp_v3_password = substr($_SERVER['argv'][6],0,(strlen($_SERVER['argv'][6]) - 1));

#
# 7th parameter is $snmp_v3_authproto
#
$snmp_v3_authproto = substr($_SERVER['argv'][7],0,(strlen($_SERVER['argv'][7]) - 1));

#
# 8th parameter is $snmp_v3_privpass
#
$snmp_v3_privpass = substr($_SERVER['argv'][8],0,(strlen($_SERVER['argv'][8]) - 1));

#
# 9th parameter is $snmp_v3_privproto
#
$snmp_v3_privproto = substr($_SERVER['argv'][9],0,(strlen($_SERVER['argv'][9]) - 1));

#
# 10th parameter is $snmp_v3_context
#
$snmp_v3_context = substr($_SERVER['argv'][10],0,(strlen($_SERVER['argv'][10]) - 1));

#
# 11th parameter is always $snmp_port
#
$snmp_port = substr($_SERVER['argv'][11],0,(strlen($_SERVER['argv'][11]) - 1));

if (is_numeric($snmp_port) == FALSE) {

	echo ("FAILED -- \"$snmp_port\" is not a numeric port number \n");

	displaySyntax();

	exit(1);
}

#
# 12th parameter is always $snmp_timeout
#
$snmp_timeout = substr($_SERVER['argv'][12],0,(strlen($_SERVER['argv'][12]) - 1));

if (is_numeric($snmp_timeout) == FALSE) {

	echo ("FAILED -- \"$snmp_timeout\" is not a numeric timeout value \n");

	displaySyntax();

	exit(1);
}

#
# Cacti's SNMP timeout uses milliseconds, while PHP uses the value
# from Net-SNMP's header files, which is typically microseconds.
# Normalize by multiplying the timeout parameter value by 1000.
#
$snmp_timeout = ($snmp_timeout * 1000);

#
# 13th parameter is always $cacti_request
#
# "query" and "get" have additional parameters
#
$cacti_request = strtolower($_SERVER['argv'][13]);

if (($cacti_request == "query") ||
	($cacti_request == "get")) {

	if (isset($_SERVER['argv'][14]) === FALSE) {

		echo ("FAILED -- incomplete cacti request\n");

		displaySyntax();

		exit(1);
	}

	else {
		$cacti_request_type = strtolower($_SERVER['argv'][14]);

		if (($cacti_request_type != "sensordevice") &&
			($cacti_request_type != "sensorname") &&
			($cacti_request_type != "sensorreading")) {

			echo ("FAILED -- unknown cacti request type\n");

			displaySyntax();

			exit(1);
		}
	}

	if ($cacti_request == "get") {

		if (isset($_SERVER['argv'][15]) === FALSE) {

			echo ("FAILED -- incomplete cacti request\n");

			displaySyntax();

			exit(1);
		}

		else {
			$cacti_request_index = $_SERVER['argv'][15];

			if (($cacti_request_index == "") ||
				($cacti_request_index < 0)) {

				echo ("FAILED -- unknown cacti index value\n");

				displaySyntax();

				exit(1);
			}
		}
	}
}

elseif ($cacti_request !== "index") {

	echo ("FAILED -- unknown cacti request\n");

	displaySyntax();

	exit(1);
}

#
# define a nested array of the OIDs and their labels
#
$oid = array (
	array ("sensorType", ".1.3.6.1.4.1.9600.1.10.6.1.3"),
	array ("sensorName", ".1.3.6.1.4.1.9600.1.10.6.1.2"),
	array ("sensorReading", ".1.3.6.1.4.1.9600.1.10.6.1.6"));

#
# create the snmp_arguments array according to the version of Cacti
# this array is used as the argument list for all SNMP function calls
#
if (substr($config["cacti_version"],0,5) == "0.8.6") {
	$snmp_arguments = array(
		$snmp_hostname,
		$snmp_community,
		$oid,
		$snmp_version,
		$snmp_v3_username,
		$snmp_v3_password,
		$snmp_port,
		$snmp_timeout,
		"",
		"");
	}

elseif (substr($config["cacti_version"],0,5) == "0.8.7") {
	$snmp_arguments = array(
		$snmp_hostname,
		$snmp_community,
		$oid,
		$snmp_version,
		$snmp_v3_username,
		$snmp_v3_password,
		$snmp_v3_authproto,
		$snmp_v3_privpass,
		$snmp_v3_privproto,
		$snmp_v3_context,
		$snmp_port,
		$snmp_timeout,
		"",
		"");
	}

else {
	echo ("failed: unsupported Cacti version\n");

	exit(1);
}

#
# if they want all the sensors, build a list of matching types
#
if (($cacti_request == "index") || ($cacti_request == "query")) {

	#
	# get a list of all sensors by type
	#
	$snmp_arguments[2] = $oid[0][1];
	$snmp_array = call_user_func_array("cacti_snmp_walk", $snmp_arguments);

	#
	# create a local array of matching sensor types
	#
	$cmdoutputcount = 0;
	$outputcount = 0;

	while ($cmdoutputcount < count($snmp_array)) {

		#
		# grab the index OID, which will be used to close all other requests
		#
		if (preg_match('/(\d+)$/', $snmp_array[$cmdoutputcount]['oid'], $scratch) == 1) {

			#
			# only look at sensors of the requested type
			#
			if ((($snmp_array[$cmdoutputcount]['value'] == 1) &&
				($sensor_type == "temperature")) ||
				(($snmp_array[$cmdoutputcount]['value'] == 2) &&
				($sensor_type == "voltage")) ||
				(($snmp_array[$cmdoutputcount]['value'] == 3) &&
				($sensor_type == "fan"))) {

				$cmdoutput[$outputcount][0] = $scratch[1];

				#
				# fill in the sensor type while we're here
				#
				$cmdoutput[$outputcount][3] =
					($snmp_array[$cmdoutputcount]['value']);

				$outputcount++;
			}
		}

		$cmdoutputcount++;
	}

	#
	# compress the array by discarding sensors with a reading of "255."
	#
	$cmdoutputcount = 0;
	$outputcount = count($cmdoutput);

	while ($cmdoutputcount < $outputcount) {

		#
		# get the sensor reading for each entry
		#
		$snmp_arguments[2] = ($oid[2][1] . "." . $cmdoutput[$cmdoutputcount][0]);
		$scratch = call_user_func_array("cacti_snmp_get", $snmp_arguments);

		#
		# remove any surrounding spaces
		#
		$scratch = trim($scratch);

		#
		# see if its the end of the list
		#
		if ($scratch == "255.") {

			unset ($cmdoutput[$cmdoutputcount]);
		}

		#
		# set the sensor reading while we're here
		#
		else {
			#
			# thermals and fans have trailing "."
			#
			if (($sensor_type == "temperature") ||
				($sensor_type == "fan")) {

				$scratch = substr($scratch, 0, (strlen($scratch) - 1));
			}

			#
			# voltages have varying decimals
			#
			if ($sensor_type == "voltage") {

				$scratch = sprintf("%01.2f", $scratch);
			}

			$cmdoutput[$cmdoutputcount][2] = $scratch;
		}

		$cmdoutputcount++;
	}

	#
	# see if any sensors were returned
	#
	if ($outputcount == 0) {

		echo ("FAILED -- no matching sensors were found\n");

		exit(1);
	}
}

#
# they just want one sensor ("get")
#
else {
	#
	# make sure the sensor is of the requested type
	#
	$snmp_arguments[2] = ($oid[0][1] . "." . $cacti_request_index);
	$scratch = call_user_func_array("cacti_snmp_get", $snmp_arguments);

	#
	# remove any surrounding spaces
	#
	$scratch = trim($scratch);

	#
	# verify the sensor type
	#
	if ((($scratch == 1) && ($sensor_type == "temperature")) ||
		(($scratch == 2) && ($sensor_type == "voltage")) ||
		(($scratch == 3) && ($sensor_type == "fan"))) {

		#
		# fill in the sensor index and type while we're here
		#
		$cmdoutput[0][0] = $cacti_request_index;
		$cmdoutput[0][3] = $scratch;
	}

	else {
		echo ("FAILED -- no matching sensors were found\n");

		exit(1);
	}

	#
	# if the sensor reading is "255." discard the sensor
	#
	$snmp_arguments[2] = ($oid[2][1] . "." . $cmdoutput[0][0]);
	$scratch = call_user_func_array("cacti_snmp_get", $snmp_arguments);

	#
	# remove any surrounding spaces
	#
	$scratch = trim($scratch);

	if ($scratch == "255.") {

		echo ("FAILED -- no matching sensors were found\n");

		exit(1);
	}

	#
	# get the sensor reading while we're here
	#
	else {
		#
		# thermals and fans have trailing "."
		#
		if (($sensor_type == "temperature") ||
			($sensor_type == "fan")) {

			$scratch = substr($scratch, 0, (strlen($scratch) - 1));
		}

		#
		# voltages have varying decimals
		#
		if ($sensor_type == "voltage") {

			$scratch = sprintf("%01.2f", $scratch);
		}

		$cmdoutput[0][2] = $scratch;
	}
}

#
# queries and lookups require additional processing
#
if (($cacti_request == "query") || ($cacti_request == "get")) {

	#
	# they want the sensor name(s)
	#
	if ($cacti_request_type == "sensorname") {

		$cmdoutputcount = 0;

		while ($cmdoutputcount < count($cmdoutput)) {

			#
			# get the sensor name for each entry
			#
			$snmp_arguments[2] = ($oid[1][1] . "." . $cmdoutput[$cmdoutputcount][0]);
			$scratch = call_user_func_array("cacti_snmp_get", $snmp_arguments);

			#
			# remove any surrounding spaces
			#
			$scratch = trim($scratch);

			#
			# if the name is long and has spaces, trim it down
			#
			while ((strlen($scratch) > 9) && (strpos($scratch, " ") !== FALSE)) {

				$scratch = (substr($scratch,0, (strrpos($scratch, " "))));
			}

			#
			# if the name is still long, chop it manually
			#
			if (strlen($scratch) > 12) {

				$scratch = (substr($scratch,0,12));
			}

			$cmdoutput[$cmdoutputcount][1] = $scratch;

			$cmdoutputcount++;
		}
	}
}

#
# generate output
#
$outputcount = 0;

while ($outputcount < count($cmdoutput)) {

	switch ($cacti_request) {

	#
	# if they want an index, dump a list of numbers
	#
	case "index":

		echo ($cmdoutput[($outputcount)][0] . "\n");

		break;

	#
	# if they want query data, give them the type they want
	#
	case "query":

		switch ($cacti_request_type) {

		#
		# they want the index numbers
		#
		case "sensordevice":

			echo ($cmdoutput[($outputcount)][0] . ":" .
				$cmdoutput[($outputcount)][0] . "\n");

			break;

		#
		# they want the sensor name
		#
		case "sensorname":

			echo ($cmdoutput[($outputcount)][0] . ":" .
				$cmdoutput[($outputcount)][1] . "\n");

			break;

		#
		# they want the sensor reading
		#
		case "sensorreading":

			echo ($cmdoutput[($outputcount)][0] . ":" .
				$cmdoutput[($outputcount)][2] . "\n");

			break;
		}

		break;

	#
	# if they want explicit data, give them the type they want
	#
	case "get":

		switch ($cacti_request_type) {

		#
		# they want the index number
		#
		case "sensordevice":

			echo ($cmdoutput[($outputcount)][0] . "\n");

			break;

		#
		# they want the sensor name
		#
		case "sensorname":

			echo ($cmdoutput[($outputcount)][1] . "\n");

			break;

		#
		# they want the sensor reading
		#
		case "sensorreading":

			echo ($cmdoutput[($outputcount)][2] . "\n");

			break;
		}

		break;
	}

	$outputcount++;
}

exit(0);

function displaySyntax() {

	#
	# display the syntax
	#
	echo ("syntax: " . $_SERVER['argv'][0] . " (FAN|TEMPERATURE|VOLTAGE) hostname,\n" .
	"   [<snmp_community>], <snmp_version>, [<snmp3_username>], [<snmp3_password>],\n" .
	"   [<snmp3_auth_protocol>], [<snmp3_priv_password>], [<snmp3_priv_protocol>],\n" .
	"   [<snmp3_context>], [<snmp_port>], <snmp_timeout>,\n" .
	"   (index|query <fieldname>|get <fieldname> <index>)\n");
}

?>
